import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { FileText, Upload, Calendar, CheckCircle } from 'lucide-react';
import { toast } from 'sonner';

export default function PaperSubmission() {
  const [formData, setFormData] = useState({
    title: '',
    authors: '',
    affiliation: '',
    email: '',
    phone: '',
    country: '',
    category: '',
    abstract: '',
    keywords: '',
    hasEthicsApproval: false,
    agreeToTerms: false
  });

  const [isSubmitted, setIsSubmitted] = useState(false);

  const handleInputChange = (field: string, value: string | boolean) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    // Basic validation
    if (!formData.title || !formData.authors || !formData.email || !formData.abstract) {
      toast.error('Please fill in all required fields');
      return;
    }

    if (!formData.agreeToTerms) {
      toast.error('Please agree to the terms and conditions');
      return;
    }

    // Simulate submission
    toast.success('Paper submitted successfully!');
    setIsSubmitted(true);
  };

  const categories = [
    'AI Policy and Governance',
    'Digital Transformation in Government',
    'Higher Education Partnerships',
    'Smart Cities and Urban Governance',
    'Digital Rights and Ethics',
    'E-Government Services',
    'Data Analytics in Public Sector',
    'Cybersecurity in Government',
    'Digital Inclusion and Accessibility',
    'Innovation in Public Administration'
  ];

  if (isSubmitted) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4 flex items-center justify-center">
        <Card className="max-w-2xl w-full border-green-200">
          <CardContent className="p-12 text-center">
            <CheckCircle className="w-16 h-16 text-green-600 mx-auto mb-6" />
            <h2 className="text-3xl font-bold text-green-800 mb-4">Submission Successful!</h2>
            <p className="text-lg text-gray-700 mb-6">
              Thank you for submitting your paper. You will receive a confirmation email shortly.
            </p>
            <div className="bg-green-50 p-6 rounded-lg mb-6">
              <h3 className="font-semibold text-green-800 mb-2">What happens next?</h3>
              <ul className="text-left text-green-700 space-y-2">
                <li>• Your paper will undergo peer review</li>
                <li>• Review results will be communicated by March 15, 2025</li>
                <li>• Accepted papers will be scheduled for presentation</li>
                <li>• Camera-ready versions due by March 30, 2025</li>
              </ul>
            </div>
            <Button onClick={() => setIsSubmitted(false)} className="bg-blue-900 hover:bg-blue-800">
              Submit Another Paper
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4">
      <div className="max-w-4xl mx-auto">
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-blue-900 mb-6">Paper Submission</h1>
          <p className="text-xl text-gray-700 max-w-3xl mx-auto">
            Submit your research paper for consideration at the IHEDG 2025 conference. 
            Share your insights on AI, digital governance, and higher education.
          </p>
        </div>

        {/* Submission Guidelines */}
        <Card className="border-blue-200 mb-8">
          <CardHeader>
            <CardTitle className="text-2xl text-blue-900 flex items-center gap-2">
              <FileText className="w-6 h-6" />
              Submission Guidelines
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-2 gap-6">
              <div>
                <h3 className="font-semibold text-blue-800 mb-3">Paper Requirements</h3>
                <ul className="space-y-2 text-gray-700">
                  <li>• Length: 6-8 pages (including references)</li>
                  <li>• Format: IEEE conference template</li>
                  <li>• Language: English</li>
                  <li>• File format: PDF only</li>
                  <li>• Font: Times New Roman, 10pt</li>
                </ul>
              </div>
              <div>
                <h3 className="font-semibold text-blue-800 mb-3">Important Dates</h3>
                <ul className="space-y-2 text-gray-700">
                  <li>• <strong>Submission Deadline:</strong> February 28, 2025</li>
                  <li>• <strong>Review Notification:</strong> March 15, 2025</li>
                  <li>• <strong>Camera-Ready:</strong> March 30, 2025</li>
                  <li>• <strong>Conference:</strong> April 8-10, 2025</li>
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Submission Form */}
        <Card className="border-blue-200">
          <CardHeader>
            <CardTitle className="text-2xl text-blue-900">Submit Your Paper</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Paper Information */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Paper Information</h3>
                
                <div>
                  <Label htmlFor="title">Paper Title *</Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => handleInputChange('title', e.target.value)}
                    placeholder="Enter your paper title"
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="category">Research Category *</Label>
                  <Select value={formData.category} onValueChange={(value) => handleInputChange('category', value)}>
                    <SelectTrigger className="mt-1">
                      <SelectValue placeholder="Select research category" />
                    </SelectTrigger>
                    <SelectContent>
                      {categories.map((category) => (
                        <SelectItem key={category} value={category}>
                          {category}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="abstract">Abstract *</Label>
                  <Textarea
                    id="abstract"
                    value={formData.abstract}
                    onChange={(e) => handleInputChange('abstract', e.target.value)}
                    placeholder="Enter your paper abstract (250-300 words)"
                    className="mt-1 min-h-32"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="keywords">Keywords</Label>
                  <Input
                    id="keywords"
                    value={formData.keywords}
                    onChange={(e) => handleInputChange('keywords', e.target.value)}
                    placeholder="Enter 5-7 keywords separated by commas"
                    className="mt-1"
                  />
                </div>
              </div>

              {/* Author Information */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Author Information</h3>
                
                <div>
                  <Label htmlFor="authors">Author(s) *</Label>
                  <Input
                    id="authors"
                    value={formData.authors}
                    onChange={(e) => handleInputChange('authors', e.target.value)}
                    placeholder="Full names of all authors"
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="affiliation">Institutional Affiliation *</Label>
                  <Input
                    id="affiliation"
                    value={formData.affiliation}
                    onChange={(e) => handleInputChange('affiliation', e.target.value)}
                    placeholder="University/Organization name"
                    className="mt-1"
                    required
                  />
                </div>

                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="email">Email Address *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      placeholder="corresponding.author@email.com"
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="phone">Phone Number</Label>
                    <Input
                      id="phone"
                      value={formData.phone}
                      onChange={(e) => handleInputChange('phone', e.target.value)}
                      placeholder="+1234567890"
                      className="mt-1"
                    />
                  </div>
                </div>

                <div>
                  <Label htmlFor="country">Country *</Label>
                  <Input
                    id="country"
                    value={formData.country}
                    onChange={(e) => handleInputChange('country', e.target.value)}
                    placeholder="Your country"
                    className="mt-1"
                    required
                  />
                </div>
              </div>

              {/* File Upload */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">File Upload</h3>
                <div className="border-2 border-dashed border-blue-300 rounded-lg p-8 text-center">
                  <Upload className="w-12 h-12 text-blue-400 mx-auto mb-4" />
                  <p className="text-gray-600 mb-2">Upload your paper (PDF format, max 10MB)</p>
                  <Button type="button" variant="outline" className="border-blue-300 text-blue-700">
                    Choose File
                  </Button>
                </div>
              </div>

              {/* Checkboxes */}
              <div className="space-y-4">
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="ethics"
                    checked={formData.hasEthicsApproval}
                    onCheckedChange={(checked) => handleInputChange('hasEthicsApproval', checked as boolean)}
                  />
                  <Label htmlFor="ethics" className="text-sm">
                    This research has received appropriate ethics approval (if applicable)
                  </Label>
                </div>

                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="terms"
                    checked={formData.agreeToTerms}
                    onCheckedChange={(checked) => handleInputChange('agreeToTerms', checked as boolean)}
                  />
                  <Label htmlFor="terms" className="text-sm">
                    I agree to the conference terms and conditions, and confirm that this work is original *
                  </Label>
                </div>
              </div>

              {/* Submit Button */}
              <div className="flex justify-center pt-6">
                <Button type="submit" size="lg" className="bg-blue-900 hover:bg-blue-800 px-8">
                  Submit Paper
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>

        {/* Contact Information */}
        <Card className="border-blue-200 mt-8">
          <CardContent className="p-6">
            <h3 className="text-lg font-semibold text-blue-800 mb-4">Need Help?</h3>
            <p className="text-gray-700">
              For questions about paper submission, please contact our review committee at{' '}
              <a href="mailto:papers@ihedg2025.org" className="text-blue-600 hover:underline">
                papers@ihedg2025.org
              </a>
            </p>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}