#!/bin/bash

# IHEDG 2025 Conference Website Deployment Script
# This script automates the deployment process for VPS servers

set -e

echo "🚀 Starting IHEDG 2025 Conference Website Deployment..."

# Configuration
APP_NAME="ihedg-conference"
DOMAIN="your-domain.com"  # Replace with your actual domain
EMAIL="admin@your-domain.com"  # Replace with your email for SSL certificates

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if running as root
if [[ $EUID -eq 0 ]]; then
   print_error "This script should not be run as root for security reasons"
   exit 1
fi

# Check if Docker is installed
if ! command -v docker &> /dev/null; then
    print_status "Docker not found. Installing Docker..."
    curl -fsSL https://get.docker.com -o get-docker.sh
    sudo sh get-docker.sh
    sudo usermod -aG docker $USER
    print_success "Docker installed successfully"
else
    print_success "Docker is already installed"
fi

# Check if Docker Compose is installed
if ! command -v docker-compose &> /dev/null; then
    print_status "Docker Compose not found. Installing Docker Compose..."
    sudo curl -L "https://github.com/docker/compose/releases/download/v2.20.0/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
    sudo chmod +x /usr/local/bin/docker-compose
    print_success "Docker Compose installed successfully"
else
    print_success "Docker Compose is already installed"
fi

# Create application directory
APP_DIR="/opt/$APP_NAME"
print_status "Creating application directory at $APP_DIR"
sudo mkdir -p $APP_DIR
sudo chown $USER:$USER $APP_DIR

# Copy application files
print_status "Copying application files..."
cp -r . $APP_DIR/
cd $APP_DIR

# Set proper permissions
chmod +x deploy.sh

# Build and start the application
print_status "Building Docker image..."
docker-compose build

print_status "Starting the application..."
docker-compose up -d

# Wait for application to start
print_status "Waiting for application to start..."
sleep 10

# Check if application is running
if docker-compose ps | grep -q "Up"; then
    print_success "Application is running successfully!"
else
    print_error "Application failed to start. Check logs with: docker-compose logs"
    exit 1
fi

# Setup SSL with Let's Encrypt (optional)
setup_ssl() {
    print_status "Setting up SSL certificate with Let's Encrypt..."
    
    # Install certbot
    if ! command -v certbot &> /dev/null; then
        sudo apt-get update
        sudo apt-get install -y certbot python3-certbot-nginx
    fi
    
    # Stop nginx temporarily
    docker-compose stop
    
    # Get SSL certificate
    sudo certbot certonly --standalone -d $DOMAIN --email $EMAIL --agree-tos --non-interactive
    
    # Update nginx config for SSL
    sed -i "s/yourdomain.com/$DOMAIN/g" nginx.conf
    sed -i 's/# server {/server {/g' nginx.conf
    sed -i 's/# }/}/g' nginx.conf
    
    # Restart with SSL
    docker-compose up -d
    
    print_success "SSL certificate installed successfully!"
}

# Setup firewall
setup_firewall() {
    print_status "Configuring firewall..."
    
    if command -v ufw &> /dev/null; then
        sudo ufw allow 22/tcp
        sudo ufw allow 80/tcp
        sudo ufw allow 443/tcp
        sudo ufw --force enable
        print_success "Firewall configured successfully!"
    else
        print_warning "UFW not found. Please configure firewall manually."
    fi
}

# Setup monitoring (optional)
setup_monitoring() {
    print_status "Setting up basic monitoring..."
    
    # Create monitoring script
    cat > monitor.sh << 'EOF'
#!/bin/bash
# Simple monitoring script for IHEDG Conference website

LOG_FILE="/var/log/ihedg-monitor.log"
TIMESTAMP=$(date '+%Y-%m-%d %H:%M:%S')

# Check if containers are running
if ! docker-compose ps | grep -q "Up"; then
    echo "[$TIMESTAMP] ERROR: Application containers are down" >> $LOG_FILE
    docker-compose up -d
    echo "[$TIMESTAMP] INFO: Attempted to restart containers" >> $LOG_FILE
fi

# Check disk space
DISK_USAGE=$(df / | awk 'NR==2 {print $5}' | sed 's/%//')
if [ $DISK_USAGE -gt 80 ]; then
    echo "[$TIMESTAMP] WARNING: Disk usage is at ${DISK_USAGE}%" >> $LOG_FILE
fi

# Check memory usage
MEM_USAGE=$(free | awk 'NR==2{printf "%.2f", $3*100/$2}')
if (( $(echo "$MEM_USAGE > 80" | bc -l) )); then
    echo "[$TIMESTAMP] WARNING: Memory usage is at ${MEM_USAGE}%" >> $LOG_FILE
fi

echo "[$TIMESTAMP] INFO: Health check completed" >> $LOG_FILE
EOF

    chmod +x monitor.sh
    
    # Add to crontab
    (crontab -l 2>/dev/null; echo "*/5 * * * * $APP_DIR/monitor.sh") | crontab -
    
    print_success "Monitoring setup completed!"
}

# Main deployment steps
print_status "=== IHEDG 2025 Conference Website Deployment ==="
print_status "Application URL: http://$(curl -s ifconfig.me)"

# Ask user for additional setup
echo ""
read -p "Do you want to setup SSL certificate? (y/n): " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    read -p "Enter your domain name: " DOMAIN
    read -p "Enter your email for SSL certificate: " EMAIL
    setup_ssl
fi

echo ""
read -p "Do you want to setup firewall? (y/n): " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    setup_firewall
fi

echo ""
read -p "Do you want to setup monitoring? (y/n): " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    setup_monitoring
fi

# Final status
print_success "=== Deployment Completed Successfully! ==="
echo ""
echo "📊 Application Status:"
docker-compose ps
echo ""
echo "🌐 Access your website at:"
echo "   HTTP: http://$(curl -s ifconfig.me)"
if [[ -n "$DOMAIN" ]]; then
    echo "   HTTPS: https://$DOMAIN"
fi
echo ""
echo "📋 Useful Commands:"
echo "   View logs: docker-compose logs -f"
echo "   Restart: docker-compose restart"
echo "   Stop: docker-compose down"
echo "   Update: git pull && docker-compose build && docker-compose up -d"
echo ""
echo "📁 Application directory: $APP_DIR"
echo "📝 Configuration files:"
echo "   - Docker Compose: docker-compose.yml"
echo "   - Nginx Config: nginx.conf"
echo "   - SSL Certificates: /etc/letsencrypt/live/$DOMAIN/"
echo ""
print_success "IHEDG 2025 Conference Website is now live! 🎉"