# IHEDG 2025 Conference Website - Deployment Guide

## Overview
This deployment package contains everything needed to deploy the IHEDG 2025 Conference website on a VPS server using Docker and Nginx.

## Quick Start

### Prerequisites
- Ubuntu 20.04+ or CentOS 8+ VPS server
- Root or sudo access
- Domain name pointed to your server (optional, for SSL)
- At least 2GB RAM and 20GB disk space

### One-Command Deployment
```bash
chmod +x deploy.sh && ./deploy.sh
```

## Manual Deployment Steps

### 1. Server Preparation
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install required packages
sudo apt install -y curl wget git unzip
```

### 2. Install Docker
```bash
curl -fsSL https://get.docker.com -o get-docker.sh
sudo sh get-docker.sh
sudo usermod -aG docker $USER
```

### 3. Install Docker Compose
```bash
sudo curl -L "https://github.com/docker/compose/releases/download/v2.20.0/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
sudo chmod +x /usr/local/bin/docker-compose
```

### 4. Deploy Application
```bash
# Clone or upload your files to /opt/ihedg-conference
sudo mkdir -p /opt/ihedg-conference
cd /opt/ihedg-conference

# Build and start
docker-compose build
docker-compose up -d
```

## Configuration Files

### Docker Compose (docker-compose.yml)
- Main orchestration file
- Configures web server and networking
- Handles port mapping and volumes

### Nginx Configuration (nginx.conf)
- Web server configuration
- Handles static file serving
- Includes security headers
- Supports client-side routing

### Dockerfile
- Multi-stage build process
- Optimized for production
- Minimal Alpine Linux base

## SSL/HTTPS Setup

### Using Let's Encrypt (Recommended)
```bash
# Install certbot
sudo apt install certbot python3-certbot-nginx

# Get certificate
sudo certbot certonly --standalone -d yourdomain.com

# Update nginx.conf with your domain
# Uncomment SSL server block
# Restart containers
docker-compose restart
```

### Using Custom SSL Certificate
1. Place certificate files in `/etc/ssl/certs/`
2. Update nginx.conf with certificate paths
3. Restart containers

## Monitoring and Maintenance

### Health Checks
```bash
# Check container status
docker-compose ps

# View logs
docker-compose logs -f

# Check resource usage
docker stats
```

### Backup Strategy
```bash
# Backup application data
tar -czf ihedg-backup-$(date +%Y%m%d).tar.gz /opt/ihedg-conference

# Backup SSL certificates
sudo tar -czf ssl-backup-$(date +%Y%m%d).tar.gz /etc/letsencrypt
```

### Updates
```bash
# Pull latest changes
git pull

# Rebuild and restart
docker-compose build
docker-compose up -d
```

## Security Considerations

### Firewall Configuration
```bash
# Using UFW
sudo ufw allow 22/tcp  # SSH
sudo ufw allow 80/tcp  # HTTP
sudo ufw allow 443/tcp # HTTPS
sudo ufw enable
```

### Security Headers
The nginx.conf includes:
- X-Frame-Options
- X-XSS-Protection
- X-Content-Type-Options
- Content-Security-Policy
- HSTS (when SSL is enabled)

## Troubleshooting

### Common Issues

#### Port Already in Use
```bash
# Check what's using port 80
sudo netstat -tulpn | grep :80

# Stop conflicting service
sudo systemctl stop apache2  # or nginx
```

#### Container Won't Start
```bash
# Check logs
docker-compose logs

# Check disk space
df -h

# Check memory
free -h
```

#### SSL Certificate Issues
```bash
# Test certificate
sudo certbot certificates

# Renew certificate
sudo certbot renew --dry-run
```

### Performance Optimization

#### Enable Gzip Compression
Already configured in nginx.conf

#### Static Asset Caching
Configured with 1-year cache for static assets

#### Resource Limits
Add to docker-compose.yml:
```yaml
deploy:
  resources:
    limits:
      memory: 512M
      cpus: '0.5'
```

## Scaling and Load Balancing

### Multiple Instances
```yaml
# In docker-compose.yml
services:
  ihedg-conference:
    scale: 3
```

### Load Balancer Setup
Use nginx upstream configuration or external load balancer

## Monitoring Setup

### Basic Monitoring Script
Included in deployment: `monitor.sh`
- Checks container health
- Monitors disk usage
- Monitors memory usage
- Logs to `/var/log/ihedg-monitor.log`

### Advanced Monitoring
Consider implementing:
- Prometheus + Grafana
- ELK Stack (Elasticsearch, Logstash, Kibana)
- Uptime monitoring services

## Support and Maintenance

### Log Locations
- Application logs: `docker-compose logs`
- Nginx access logs: `/var/log/nginx/access.log`
- Nginx error logs: `/var/log/nginx/error.log`
- System logs: `/var/log/syslog`

### Regular Maintenance Tasks
1. Update system packages monthly
2. Renew SSL certificates (automated with certbot)
3. Monitor disk space and clean old logs
4. Review security updates
5. Backup configuration and data

### Emergency Procedures
```bash
# Quick restart
docker-compose restart

# Full rebuild
docker-compose down
docker-compose build --no-cache
docker-compose up -d

# Rollback (if you have backups)
docker-compose down
# Restore from backup
docker-compose up -d
```

## Contact Information
For deployment support or issues:
- Technical Support: tech@ihedg2025.org
- Emergency Contact: +27 21 406 1911

---

**IHEDG 2025 Conference Website**  
*Intelligence and the Role of Higher Education in Enabling Digital Government*  
April 8-10, 2025 | South Africa