# IHEDG 2025 Conference - TMD Cloud Deployment Guide

## 🚀 TMD Cloud Optimized Deployment

This guide provides comprehensive instructions for deploying the IHEDG 2025 Conference website on TMD Cloud VPS servers with enhanced performance, security, and monitoring.

## 🏗️ TMD Cloud Architecture

```
┌─────────────────┐    ┌──────────────────┐    ┌─────────────────┐
│   Load Balancer │────│   TMD Cloud VPS  │────│   SSL/TLS       │
│   (Optional)    │    │   Ubuntu 20.04+  │    │   Let's Encrypt │
└─────────────────┘    └──────────────────┘    └─────────────────┘
                              │
                    ┌─────────┴─────────┐
                    │                   │
              ┌─────────────┐    ┌─────────────┐
              │   Docker    │    │  Monitoring │
              │  Container  │    │  Dashboard  │
              └─────────────┘    └─────────────┘
```

## 🚀 Quick TMD Cloud Deployment

### Prerequisites
- TMD Cloud VPS server (2GB+ RAM, 20GB+ storage)
- Ubuntu 20.04+ or CentOS 8+
- Root or sudo access
- Domain name pointed to your TMD Cloud server IP

### One-Command Deployment
```bash
# Download and run TMD Cloud deployment script
wget https://your-repo/deploy-tmd.sh
chmod +x deploy-tmd.sh
DOMAIN=your-domain.com EMAIL=your-email@domain.com ./deploy-tmd.sh
```

## 📋 Manual TMD Cloud Setup

### 1. Server Preparation
```bash
# Update TMD Cloud VPS
sudo apt update && sudo apt upgrade -y

# Install essential packages
sudo apt install -y curl wget git unzip htop iotop net-tools ufw fail2ban

# Set timezone
sudo timedatectl set-timezone UTC
```

### 2. Security Configuration
```bash
# Configure UFW firewall
sudo ufw --force reset
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow 22/tcp
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp
sudo ufw --force enable

# Configure fail2ban
sudo systemctl enable fail2ban
sudo systemctl start fail2ban
```

### 3. Docker Installation
```bash
# Install Docker CE
curl -fsSL https://get.docker.com -o get-docker.sh
sudo sh get-docker.sh
sudo usermod -aG docker $USER

# Install Docker Compose
sudo curl -L "https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
sudo chmod +x /usr/local/bin/docker-compose
```

### 4. Application Deployment
```bash
# Create application directory
sudo mkdir -p /opt/ihedg-conference
sudo chown $USER:$USER /opt/ihedg-conference
cd /opt/ihedg-conference

# Deploy with TMD Cloud configuration
docker compose -f docker-compose-tmd.yml build
docker compose -f docker-compose-tmd.yml up -d
```

## 🔧 TMD Cloud Configuration Files

### Docker Compose (docker-compose-tmd.yml)
- Optimized for TMD Cloud infrastructure
- Health checks and resource limits
- TMD Cloud specific labels and networks
- Monitoring integration

### Nginx Configuration (nginx-tmd.conf)
- TMD Cloud performance optimizations
- Enhanced security headers
- Rate limiting and caching
- SSL/TLS configuration

### Deployment Script (deploy-tmd.sh)
- Automated TMD Cloud setup
- Security hardening
- SSL certificate automation
- Monitoring dashboard setup

## 🔒 TMD Cloud Security Features

### Firewall Configuration
```bash
# TMD Cloud optimized firewall rules
sudo ufw allow from 10.0.0.0/8 to any port 22    # TMD internal network
sudo ufw allow from 172.16.0.0/12 to any port 22  # TMD management
sudo ufw deny 22/tcp                               # Block external SSH
```

### SSL/TLS Setup
```bash
# Automatic SSL with Let's Encrypt
sudo certbot certonly --standalone -d your-domain.com
sudo certbot renew --dry-run  # Test auto-renewal
```

### Security Headers
- X-Frame-Options: SAMEORIGIN
- X-XSS-Protection: 1; mode=block
- X-Content-Type-Options: nosniff
- Strict-Transport-Security (HSTS)
- Content-Security-Policy

## 📊 TMD Cloud Monitoring

### Health Checks
```bash
# Application health
curl -f http://your-domain.com/health

# Container status
docker compose -f docker-compose-tmd.yml ps

# Resource usage
docker stats
```

### Monitoring Dashboard
Access at: `http://your-domain.com:8080`
- Real-time service status
- TMD Cloud region information
- Performance metrics
- Error logs

### Log Management
```bash
# View application logs
docker compose -f docker-compose-tmd.yml logs -f

# Nginx access logs
docker exec -it ihedg-conference-tmd tail -f /var/log/nginx/access.log

# System logs
sudo journalctl -u docker -f
```

## ⚡ TMD Cloud Performance Optimization

### Caching Strategy
- Static assets: 1 year cache
- HTML files: 1 hour cache
- API responses: No cache
- Gzip compression enabled

### Resource Limits
```yaml
deploy:
  resources:
    limits:
      memory: 512M
      cpus: '1.0'
    reservations:
      memory: 256M
      cpus: '0.5'
```

### CDN Integration (Optional)
```bash
# Configure TMD Cloud CDN
# Add CDN endpoints to nginx configuration
# Update asset URLs to use CDN
```

## 🔄 TMD Cloud Maintenance

### Regular Updates
```bash
# Update system packages
sudo apt update && sudo apt upgrade -y

# Update Docker images
docker compose -f docker-compose-tmd.yml pull
docker compose -f docker-compose-tmd.yml up -d

# Clean up unused resources
docker system prune -f
```

### Backup Strategy
```bash
# Backup application data
tar -czf ihedg-backup-$(date +%Y%m%d).tar.gz /opt/ihedg-conference

# Backup SSL certificates
sudo tar -czf ssl-backup-$(date +%Y%m%d).tar.gz /etc/letsencrypt

# Upload to TMD Cloud storage (if available)
# tmd-cli storage upload ihedg-backup-$(date +%Y%m%d).tar.gz
```

### Scaling on TMD Cloud
```bash
# Horizontal scaling
docker compose -f docker-compose-tmd.yml up -d --scale ihedg-conference=3

# Load balancer configuration (if using TMD Cloud LB)
# Configure TMD Cloud Load Balancer to distribute traffic
```

## 🚨 TMD Cloud Troubleshooting

### Common Issues

#### Port Already in Use
```bash
# Check what's using ports
sudo netstat -tulpn | grep :80
sudo netstat -tulpn | grep :443

# Stop conflicting services
sudo systemctl stop apache2 nginx
```

#### Container Won't Start
```bash
# Check logs
docker compose -f docker-compose-tmd.yml logs ihedg-conference

# Check resources
df -h  # Disk space
free -h  # Memory usage
```

#### SSL Certificate Issues
```bash
# Check certificate status
sudo certbot certificates

# Test renewal
sudo certbot renew --dry-run

# Manual renewal
sudo certbot renew --force-renewal
```

### Performance Issues
```bash
# Monitor resource usage
htop
iotop
docker stats

# Check nginx performance
curl -w "@curl-format.txt" -o /dev/null -s http://your-domain.com/
```

## 📞 TMD Cloud Support

### Emergency Procedures
```bash
# Quick restart
docker compose -f docker-compose-tmd.yml restart

# Full rebuild
docker compose -f docker-compose-tmd.yml down
docker compose -f docker-compose-tmd.yml build --no-cache
docker compose -f docker-compose-tmd.yml up -d

# Rollback to previous version
docker compose -f docker-compose-tmd.yml down
# Restore from backup
docker compose -f docker-compose-tmd.yml up -d
```

### Contact Information
- **TMD Cloud Support**: support@tmdcloud.com
- **Technical Issues**: tech@ihedg2025.org
- **Emergency**: +1-XXX-XXX-XXXX

## 🎯 TMD Cloud Best Practices

1. **Regular Monitoring**: Check health endpoints daily
2. **Security Updates**: Apply patches monthly
3. **Backup Verification**: Test backups weekly
4. **Performance Review**: Monitor metrics continuously
5. **SSL Renewal**: Automated with certbot
6. **Log Rotation**: Configured in Docker daemon
7. **Resource Monitoring**: Set up alerts for high usage

---

**IHEDG 2025 Conference on TMD Cloud**  
*Optimized for Performance, Security, and Reliability*  
April 8-10, 2025 | South Africa

For more information about TMD Cloud services, visit: https://tmdcloud.com