#!/bin/bash

# IHEDG 2025 Conference Website - TMD Cloud Deployment Script
# Optimized for TMD Cloud VPS servers with enhanced security and performance

set -e

echo "🚀 Starting IHEDG 2025 Conference Website Deployment on TMD Cloud..."

# Configuration for TMD Cloud
APP_NAME="ihedg-conference"
DOMAIN="${DOMAIN:-your-domain.com}"  # Set via environment variable
EMAIL="${EMAIL:-admin@your-domain.com}"
TMD_REGION="${TMD_REGION:-us-east-1}"  # TMD Cloud region

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[TMD-INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[TMD-SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[TMD-WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[TMD-ERROR]${NC} $1"
}

print_tmd() {
    echo -e "${PURPLE}[TMD-CLOUD]${NC} $1"
}

# Check if running as root
if [[ $EUID -eq 0 ]]; then
   print_error "This script should not be run as root for security reasons"
   exit 1
fi

# TMD Cloud specific system preparation
prepare_tmd_system() {
    print_tmd "Preparing TMD Cloud VPS system..."
    
    # Update system packages
    sudo apt update && sudo apt upgrade -y
    
    # Install TMD Cloud monitoring agent (if available)
    if command -v tmd-agent &> /dev/null; then
        print_success "TMD Cloud monitoring agent already installed"
    else
        print_warning "TMD Cloud monitoring agent not found. Install manually if needed."
    fi
    
    # Install essential packages for TMD Cloud
    sudo apt install -y curl wget git unzip htop iotop net-tools ufw fail2ban
    
    # Configure timezone for TMD Cloud region
    sudo timedatectl set-timezone UTC
    
    print_success "TMD Cloud system preparation completed"
}

# Enhanced security configuration for TMD Cloud
configure_tmd_security() {
    print_tmd "Configuring enhanced security for TMD Cloud..."
    
    # Configure UFW firewall
    sudo ufw --force reset
    sudo ufw default deny incoming
    sudo ufw default allow outgoing
    sudo ufw allow 22/tcp comment 'SSH'
    sudo ufw allow 80/tcp comment 'HTTP'
    sudo ufw allow 443/tcp comment 'HTTPS'
    sudo ufw --force enable
    
    # Configure fail2ban
    sudo systemctl enable fail2ban
    sudo systemctl start fail2ban
    
    # Create fail2ban configuration for nginx
    sudo tee /etc/fail2ban/jail.local > /dev/null <<EOF
[DEFAULT]
bantime = 3600
findtime = 600
maxretry = 5

[nginx-http-auth]
enabled = true

[nginx-noscript]
enabled = true

[nginx-badbots]
enabled = true

[nginx-noproxy]
enabled = true
EOF
    
    sudo systemctl restart fail2ban
    
    print_success "TMD Cloud security configuration completed"
}

# Install Docker optimized for TMD Cloud
install_docker_tmd() {
    print_tmd "Installing Docker optimized for TMD Cloud..."
    
    if ! command -v docker &> /dev/null; then
        # Remove old versions
        sudo apt remove -y docker docker-engine docker.io containerd runc || true
        
        # Install Docker CE
        curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg
        echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null
        
        sudo apt update
        sudo apt install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
        
        # Add user to docker group
        sudo usermod -aG docker $USER
        
        # Configure Docker daemon for TMD Cloud
        sudo tee /etc/docker/daemon.json > /dev/null <<EOF
{
    "log-driver": "json-file",
    "log-opts": {
        "max-size": "10m",
        "max-file": "3"
    },
    "storage-driver": "overlay2",
    "live-restore": true,
    "userland-proxy": false,
    "no-new-privileges": true
}
EOF
        
        sudo systemctl enable docker
        sudo systemctl restart docker
        
        print_success "Docker installed and configured for TMD Cloud"
    else
        print_success "Docker already installed"
    fi
    
    # Install Docker Compose if not available
    if ! docker compose version &> /dev/null; then
        sudo curl -L "https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
        sudo chmod +x /usr/local/bin/docker-compose
        print_success "Docker Compose installed"
    fi
}

# Create optimized nginx configuration for TMD Cloud
create_tmd_nginx_config() {
    print_tmd "Creating optimized nginx configuration for TMD Cloud..."
    
    cat > nginx-tmd.conf <<EOF
# TMD Cloud optimized nginx configuration
server {
    listen 80;
    server_name ${DOMAIN} www.${DOMAIN};
    root /usr/share/nginx/html;
    index index.html;
    
    # TMD Cloud specific optimizations
    client_max_body_size 50M;
    client_body_timeout 60s;
    client_header_timeout 60s;
    keepalive_timeout 65s;
    send_timeout 60s;

    # Enhanced gzip compression for TMD Cloud
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_comp_level 6;
    gzip_types
        text/plain
        text/css
        text/xml
        text/javascript
        application/javascript
        application/xml+rss
        application/json
        application/xml
        image/svg+xml;

    # TMD Cloud security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;
    add_header Content-Security-Policy "default-src 'self' http: https: data: blob: 'unsafe-inline' 'unsafe-eval'" always;
    add_header X-TMD-Cloud "Powered by TMD Cloud" always;

    # Performance optimizations for TMD Cloud
    location / {
        try_files \$uri \$uri/ /index.html;
        
        # Cache static assets aggressively on TMD Cloud
        location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot|webp|avif)$ {
            expires 1y;
            add_header Cache-Control "public, immutable";
            add_header X-TMD-Cache "HIT";
        }
        
        # Cache HTML files for shorter period
        location ~* \.(html|htm)$ {
            expires 1h;
            add_header Cache-Control "public, must-revalidate";
        }
    }

    # Rate limiting for TMD Cloud
    location /api/ {
        limit_req zone=api burst=20 nodelay;
        proxy_pass http://backend:3000/;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_set_header X-TMD-Cloud-Region "${TMD_REGION}";
        proxy_cache_bypass \$http_upgrade;
    }

    # TMD Cloud health check endpoint
    location /health {
        access_log off;
        return 200 "TMD Cloud - IHEDG 2025 Conference - Healthy\n";
        add_header Content-Type text/plain;
    }

    # Error pages
    error_page 404 /index.html;
    error_page 500 502 503 504 /50x.html;
    
    location = /50x.html {
        root /usr/share/nginx/html;
    }
}

# Rate limiting zones
http {
    limit_req_zone \$binary_remote_addr zone=api:10m rate=10r/s;
    limit_req_zone \$binary_remote_addr zone=general:10m rate=30r/s;
}

# SSL configuration for TMD Cloud (uncomment after SSL setup)
# server {
#     listen 443 ssl http2;
#     server_name ${DOMAIN} www.${DOMAIN};
#     root /usr/share/nginx/html;
#     index index.html;
#
#     ssl_certificate /etc/letsencrypt/live/${DOMAIN}/fullchain.pem;
#     ssl_certificate_key /etc/letsencrypt/live/${DOMAIN}/privkey.pem;
#     ssl_session_timeout 1d;
#     ssl_session_cache shared:TMDCloudSSL:10m;
#     ssl_session_tickets off;
#
#     ssl_protocols TLSv1.2 TLSv1.3;
#     ssl_ciphers ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384;
#     ssl_prefer_server_ciphers off;
#
#     # HSTS for TMD Cloud
#     add_header Strict-Transport-Security "max-age=63072000; includeSubDomains; preload" always;
#
#     location / {
#         try_files \$uri \$uri/ /index.html;
#     }
# }
EOF

    print_success "TMD Cloud nginx configuration created"
}

# Create TMD Cloud optimized Docker Compose
create_tmd_docker_compose() {
    print_tmd "Creating TMD Cloud optimized Docker Compose configuration..."
    
    cat > docker-compose-tmd.yml <<EOF
version: '3.8'

services:
  ihedg-conference:
    build: 
      context: .
      dockerfile: Dockerfile
    container_name: ihedg-conference-tmd
    restart: unless-stopped
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./nginx-tmd.conf:/etc/nginx/conf.d/default.conf:ro
      - /etc/letsencrypt:/etc/letsencrypt:ro
      - tmd_logs:/var/log/nginx
    environment:
      - NODE_ENV=production
      - TMD_CLOUD_REGION=${TMD_REGION}
      - DOMAIN=${DOMAIN}
    networks:
      - tmd-network
    labels:
      - "com.tmd.cloud.service=ihedg-conference"
      - "com.tmd.cloud.region=${TMD_REGION}"
      - "com.tmd.cloud.domain=${DOMAIN}"
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s
    deploy:
      resources:
        limits:
          memory: 512M
          cpus: '1.0'
        reservations:
          memory: 256M
          cpus: '0.5'

  # TMD Cloud monitoring (optional)
  tmd-monitor:
    image: nginx:alpine
    container_name: tmd-monitor
    restart: unless-stopped
    ports:
      - "8080:80"
    volumes:
      - ./monitor.html:/usr/share/nginx/html/index.html:ro
    networks:
      - tmd-network
    depends_on:
      - ihedg-conference

volumes:
  tmd_logs:
    driver: local

networks:
  tmd-network:
    driver: bridge
    labels:
      - "com.tmd.cloud.network=ihedg-conference"
EOF

    print_success "TMD Cloud Docker Compose configuration created"
}

# Create TMD Cloud monitoring dashboard
create_tmd_monitoring() {
    print_tmd "Creating TMD Cloud monitoring dashboard..."
    
    cat > monitor.html <<EOF
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TMD Cloud - IHEDG 2025 Monitor</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .header { text-align: center; color: #333; border-bottom: 2px solid #007acc; padding-bottom: 10px; }
        .status { display: flex; justify-content: space-between; margin: 20px 0; }
        .metric { text-align: center; padding: 15px; background: #f8f9fa; border-radius: 5px; }
        .healthy { color: #28a745; }
        .warning { color: #ffc107; }
        .error { color: #dc3545; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 TMD Cloud - IHEDG 2025 Conference</h1>
            <p>Real-time Monitoring Dashboard</p>
        </div>
        <div class="status">
            <div class="metric">
                <h3>Service Status</h3>
                <p class="healthy">✅ Online</p>
            </div>
            <div class="metric">
                <h3>Region</h3>
                <p>${TMD_REGION}</p>
            </div>
            <div class="metric">
                <h3>Domain</h3>
                <p>${DOMAIN}</p>
            </div>
        </div>
        <div class="metric">
            <h3>Last Updated</h3>
            <p id="timestamp"></p>
        </div>
    </div>
    <script>
        document.getElementById('timestamp').textContent = new Date().toLocaleString();
        setInterval(() => {
            document.getElementById('timestamp').textContent = new Date().toLocaleString();
        }, 1000);
    </script>
</body>
</html>
EOF

    print_success "TMD Cloud monitoring dashboard created"
}

# Setup SSL with Let's Encrypt for TMD Cloud
setup_tmd_ssl() {
    print_tmd "Setting up SSL certificate with Let's Encrypt for TMD Cloud..."
    
    if [[ "$DOMAIN" == "your-domain.com" ]]; then
        print_warning "Please set your actual domain name. Skipping SSL setup."
        return
    fi
    
    # Install certbot
    if ! command -v certbot &> /dev/null; then
        sudo apt update
        sudo apt install -y certbot python3-certbot-nginx
    fi
    
    # Stop containers temporarily
    docker compose -f docker-compose-tmd.yml down || true
    
    # Get SSL certificate
    sudo certbot certonly --standalone -d $DOMAIN -d www.$DOMAIN --email $EMAIL --agree-tos --non-interactive
    
    if [ $? -eq 0 ]; then
        print_success "SSL certificate obtained successfully for TMD Cloud"
        
        # Update nginx config to enable SSL
        sed -i 's/# server {/server {/g' nginx-tmd.conf
        sed -i 's/# }/}/g' nginx-tmd.conf
        sed -i 's/# ssl_/ssl_/g' nginx-tmd.conf
        sed -i 's/# add_header Strict/add_header Strict/g' nginx-tmd.conf
        sed -i 's/# location \/ {/location \/ {/g' nginx-tmd.conf
        
        # Setup auto-renewal
        echo "0 12 * * * /usr/bin/certbot renew --quiet" | sudo crontab -
        
        print_success "SSL auto-renewal configured for TMD Cloud"
    else
        print_error "Failed to obtain SSL certificate"
    fi
}

# Deploy application to TMD Cloud
deploy_to_tmd() {
    print_tmd "Deploying IHEDG 2025 Conference to TMD Cloud..."
    
    # Create application directory
    APP_DIR="/opt/$APP_NAME"
    sudo mkdir -p $APP_DIR
    sudo chown $USER:$USER $APP_DIR
    
    # Copy files
    cp -r . $APP_DIR/
    cd $APP_DIR
    
    # Build and deploy
    docker compose -f docker-compose-tmd.yml build
    docker compose -f docker-compose-tmd.yml up -d
    
    # Wait for services to start
    print_status "Waiting for services to start on TMD Cloud..."
    sleep 15
    
    # Health check
    if curl -f http://localhost/health > /dev/null 2>&1; then
        print_success "IHEDG 2025 Conference deployed successfully on TMD Cloud!"
    else
        print_error "Deployment health check failed"
        docker compose -f docker-compose-tmd.yml logs
        exit 1
    fi
}

# Main deployment process
main() {
    print_tmd "=== TMD Cloud Deployment for IHEDG 2025 Conference ==="
    
    # Check for domain configuration
    if [[ -z "$DOMAIN" || "$DOMAIN" == "your-domain.com" ]]; then
        read -p "Enter your domain name for TMD Cloud: " DOMAIN
        export DOMAIN
    fi
    
    if [[ -z "$EMAIL" || "$EMAIL" == "admin@your-domain.com" ]]; then
        read -p "Enter your email for SSL certificates: " EMAIL
        export EMAIL
    fi
    
    # Execute deployment steps
    prepare_tmd_system
    configure_tmd_security
    install_docker_tmd
    create_tmd_nginx_config
    create_tmd_docker_compose
    create_tmd_monitoring
    deploy_to_tmd
    
    # Optional SSL setup
    echo ""
    read -p "Do you want to setup SSL certificate for TMD Cloud? (y/n): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        setup_tmd_ssl
        docker compose -f docker-compose-tmd.yml up -d
    fi
    
    # Final status
    print_success "=== TMD Cloud Deployment Completed Successfully! ==="
    echo ""
    echo "🌐 TMD Cloud Access URLs:"
    echo "   Main Site: http://$DOMAIN"
    echo "   Monitor: http://$DOMAIN:8080"
    if [[ -f "/etc/letsencrypt/live/$DOMAIN/fullchain.pem" ]]; then
        echo "   HTTPS: https://$DOMAIN"
    fi
    echo ""
    echo "📊 TMD Cloud Management:"
    echo "   View logs: docker compose -f docker-compose-tmd.yml logs -f"
    echo "   Restart: docker compose -f docker-compose-tmd.yml restart"
    echo "   Stop: docker compose -f docker-compose-tmd.yml down"
    echo "   Update: git pull && docker compose -f docker-compose-tmd.yml build && docker compose -f docker-compose-tmd.yml up -d"
    echo ""
    echo "📁 TMD Cloud Files:"
    echo "   App Directory: $APP_DIR"
    echo "   Config: docker-compose-tmd.yml"
    echo "   Nginx: nginx-tmd.conf"
    echo "   SSL: /etc/letsencrypt/live/$DOMAIN/"
    echo ""
    print_tmd "IHEDG 2025 Conference is now live on TMD Cloud! 🎉"
}

# Run main deployment
main "$@"