import { useState } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { Button } from '@/components/ui/button';
import { Sheet, SheetContent, SheetTrigger } from '@/components/ui/sheet';
import { Menu, Sparkles } from 'lucide-react';

const Navbar = () => {
  const [isOpen, setIsOpen] = useState(false);
  const location = useLocation();

  const navItems = [
    { name: 'Home', path: '/', icon: '🏠' },
    { name: 'About', path: '/about', icon: '💡' },
    { name: 'Speakers', path: '/speakers', icon: '🎤' },
    { name: 'Schedule', path: '/schedule', icon: '📅' },
    { name: 'Submit Paper', path: '/submit-paper', icon: '📝' },
    { name: 'Register', path: '/register', icon: '🎯' },
    { name: 'Contact', path: '/contact', icon: '📧' },
  ];

  const isActive = (path: string) => location.pathname === path;

  return (
    <nav className="bg-gradient-to-r from-purple-900 via-blue-900 to-indigo-900 backdrop-blur-md border-b border-white/10 sticky top-0 z-50 shadow-2xl">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between h-16">
          <div className="flex items-center">
            <Link to="/" className="flex-shrink-0 flex items-center group">
              <div className="bg-white/10 backdrop-blur-sm p-2 rounded-xl mr-3 group-hover:scale-110 transition-transform duration-300 border border-white/20">
                <img 
                  src="/logo.png" 
                  alt="IHEDG 2025 Logo" 
                  className="w-8 h-8 object-contain"
                />
              </div>
              <div>
                <h1 className="text-xl font-bold text-transparent bg-clip-text bg-gradient-to-r from-pink-400 to-violet-400">
                  IHEDG 2025
                </h1>
                <p className="text-xs text-gray-300">Digital Future</p>
              </div>
            </Link>
          </div>

          {/* Desktop Navigation */}
          <div className="hidden md:flex items-center space-x-2">
            {navItems.map((item) => (
              <Link
                key={item.name}
                to={item.path}
                className={`px-4 py-2 rounded-xl text-sm font-medium transition-all duration-300 flex items-center gap-2 ${
                  isActive(item.path)
                    ? 'bg-gradient-to-r from-pink-500 to-violet-500 text-white shadow-lg transform scale-105'
                    : 'text-gray-200 hover:bg-white/10 hover:text-white hover:scale-105'
                }`}
              >
                <span>{item.icon}</span>
                {item.name}
              </Link>
            ))}
          </div>

          {/* CTA Button */}
          <div className="hidden md:flex items-center">
            <Button asChild className="bg-gradient-to-r from-yellow-500 to-orange-500 hover:from-yellow-600 hover:to-orange-600 text-white font-bold rounded-xl shadow-lg transform hover:scale-105 transition-all duration-300">
              <Link to="/register" className="flex items-center gap-2">
                <Sparkles className="w-4 h-4" />
                Join Now
              </Link>
            </Button>
          </div>

          {/* Mobile Navigation */}
          <div className="md:hidden">
            <Sheet open={isOpen} onOpenChange={setIsOpen}>
              <SheetTrigger asChild>
                <Button variant="ghost" size="icon" className="text-white hover:bg-white/10">
                  <Menu className="h-6 w-6" />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="bg-gradient-to-b from-purple-900 to-indigo-900 border-l border-white/10">
                <div className="flex flex-col space-y-4 mt-8">
                  <div className="flex items-center mb-6">
                    <img 
                      src="/logo.png" 
                      alt="IHEDG 2025 Logo" 
                      className="w-8 h-8 object-contain mr-3"
                    />
                    <span className="text-white font-bold">IHEDG 2025</span>
                  </div>
                  {navItems.map((item) => (
                    <Link
                      key={item.name}
                      to={item.path}
                      onClick={() => setIsOpen(false)}
                      className={`px-4 py-3 rounded-xl text-sm font-medium transition-all duration-300 flex items-center gap-3 ${
                        isActive(item.path)
                          ? 'bg-gradient-to-r from-pink-500 to-violet-500 text-white shadow-lg'
                          : 'text-gray-200 hover:bg-white/10 hover:text-white'
                      }`}
                    >
                      <span className="text-lg">{item.icon}</span>
                      {item.name}
                    </Link>
                  ))}
                  <Button asChild className="bg-gradient-to-r from-yellow-500 to-orange-500 hover:from-yellow-600 hover:to-orange-600 text-white font-bold rounded-xl shadow-lg mt-4">
                    <Link to="/register" className="flex items-center gap-2">
                      <Sparkles className="w-4 h-4" />
                      Join Now
                    </Link>
                  </Button>
                </div>
              </SheetContent>
            </Sheet>
          </div>
        </div>
      </div>
    </nav>
  );
};

export default Navbar;