import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { MapPin, Phone, Mail, Clock, Building, Users } from 'lucide-react';
import { useState } from 'react';
import { toast } from 'sonner';

export default function Contact() {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    subject: '',
    message: ''
  });

  const handleInputChange = (field: string, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name || !formData.email || !formData.message) {
      toast.error('Please fill in all required fields');
      return;
    }

    toast.success('Message sent successfully! We will get back to you soon.');
    setFormData({ name: '', email: '', subject: '', message: '' });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4">
      <div className="max-w-6xl mx-auto">
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-blue-900 mb-6">Contact Us</h1>
          <p className="text-xl text-gray-700 max-w-3xl mx-auto">
            Have questions about the conference? Need assistance with registration or paper submission? 
            We're here to help. Get in touch with our organizing committee.
          </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-12">
          {/* Contact Information */}
          <div className="space-y-8">
            {/* Conference Details */}
            <Card className="border-blue-200">
              <CardHeader>
                <CardTitle className="text-xl text-blue-900 flex items-center gap-2">
                  <Building className="w-5 h-5" />
                  Conference Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-start space-x-3">
                  <MapPin className="w-5 h-5 text-blue-600 mt-1 flex-shrink-0" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Venue</h3>
                    <p className="text-gray-600">
                      University of Cape Town<br />
                      Graduate School of Business<br />
                      Breakwater Campus<br />
                      Cape Town, South Africa
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start space-x-3">
                  <Clock className="w-5 h-5 text-blue-600 mt-1 flex-shrink-0" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Conference Dates</h3>
                    <p className="text-gray-600">
                      Tuesday, April 8 - Thursday, April 10, 2025<br />
                      9:00 AM - 6:00 PM daily
                    </p>
                  </div>
                </div>

                <div className="flex items-start space-x-3">
                  <Users className="w-5 h-5 text-blue-600 mt-1 flex-shrink-0" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Expected Attendance</h3>
                    <p className="text-gray-600">
                      200 delegates from academia, government, and industry
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Contact Details */}
            <Card className="border-blue-200">
              <CardHeader>
                <CardTitle className="text-xl text-blue-900">Get in Touch</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center space-x-3">
                  <Mail className="w-5 h-5 text-blue-600" />
                  <div>
                    <h3 className="font-semibold text-gray-800">General Inquiries</h3>
                    <a href="mailto:info@ihedg2025.org" className="text-blue-600 hover:underline">
                      info@ihedg2025.org
                    </a>
                  </div>
                </div>

                <div className="flex items-center space-x-3">
                  <Mail className="w-5 h-5 text-blue-600" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Paper Submissions</h3>
                    <a href="mailto:papers@ihedg2025.org" className="text-blue-600 hover:underline">
                      papers@ihedg2025.org
                    </a>
                  </div>
                </div>

                <div className="flex items-center space-x-3">
                  <Mail className="w-5 h-5 text-blue-600" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Registration Support</h3>
                    <a href="mailto:registration@ihedg2025.org" className="text-blue-600 hover:underline">
                      registration@ihedg2025.org
                    </a>
                  </div>
                </div>

                <div className="flex items-center space-x-3">
                  <Phone className="w-5 h-5 text-blue-600" />
                  <div>
                    <h3 className="font-semibold text-gray-800">Phone Support</h3>
                    <p className="text-gray-600">+27 21 406 1911</p>
                    <p className="text-sm text-gray-500">Monday - Friday, 9:00 AM - 5:00 PM (SAST)</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Organizing Committee */}
            <Card className="border-blue-200">
              <CardHeader>
                <CardTitle className="text-xl text-blue-900">Organizing Committee</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <h3 className="font-semibold text-gray-800">Prof. Nomsa Mthembu</h3>
                  <p className="text-sm text-gray-600">Conference Chair</p>
                  <p className="text-sm text-gray-600">University of Cape Town</p>
                  <a href="mailto:nomsa.mthembu@uct.ac.za" className="text-blue-600 hover:underline text-sm">
                    nomsa.mthembu@uct.ac.za
                  </a>
                </div>

                <div>
                  <h3 className="font-semibold text-gray-800">Dr. Thabo Mokwena</h3>
                  <p className="text-sm text-gray-600">Program Committee Chair</p>
                  <p className="text-sm text-gray-600">University of the Witwatersrand</p>
                  <a href="mailto:thabo.mokwena@wits.ac.za" className="text-blue-600 hover:underline text-sm">
                    thabo.mokwena@wits.ac.za
                  </a>
                </div>

                <div>
                  <h3 className="font-semibold text-gray-800">Ms. Sarah Johnson</h3>
                  <p className="text-sm text-gray-600">Conference Coordinator</p>
                  <p className="text-sm text-gray-600">Event Management Team</p>
                  <a href="mailto:sarah.johnson@ihedg2025.org" className="text-blue-600 hover:underline text-sm">
                    sarah.johnson@ihedg2025.org
                  </a>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Contact Form */}
          <div>
            <Card className="border-blue-200">
              <CardHeader>
                <CardTitle className="text-xl text-blue-900">Send us a Message</CardTitle>
              </CardHeader>
              <CardContent>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div>
                    <Label htmlFor="name">Full Name *</Label>
                    <Input
                      id="name"
                      value={formData.name}
                      onChange={(e) => handleInputChange('name', e.target.value)}
                      placeholder="Enter your full name"
                      className="mt-1"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="email">Email Address *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      placeholder="your.email@example.com"
                      className="mt-1"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="subject">Subject</Label>
                    <Input
                      id="subject"
                      value={formData.subject}
                      onChange={(e) => handleInputChange('subject', e.target.value)}
                      placeholder="Brief subject of your inquiry"
                      className="mt-1"
                    />
                  </div>

                  <div>
                    <Label htmlFor="message">Message *</Label>
                    <Textarea
                      id="message"
                      value={formData.message}
                      onChange={(e) => handleInputChange('message', e.target.value)}
                      placeholder="Please provide details about your inquiry..."
                      className="mt-1 min-h-32"
                      required
                    />
                  </div>

                  <Button type="submit" className="w-full bg-blue-900 hover:bg-blue-800">
                    Send Message
                  </Button>
                </form>
              </CardContent>
            </Card>

            {/* Map Placeholder */}
            <Card className="border-blue-200 mt-8">
              <CardHeader>
                <CardTitle className="text-xl text-blue-900">Conference Location</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="bg-blue-100 h-64 rounded-lg flex items-center justify-center">
                  <div className="text-center text-blue-800">
                    <MapPin className="w-12 h-12 mx-auto mb-2" />
                    <p className="font-semibold">University of Cape Town</p>
                    <p className="text-sm">Graduate School of Business</p>
                    <p className="text-sm">Breakwater Campus, Cape Town</p>
                  </div>
                </div>
                <div className="mt-4 text-sm text-gray-600">
                  <p><strong>Address:</strong> Portswood Road, Green Point, Cape Town, 8005, South Africa</p>
                  <p><strong>Parking:</strong> On-campus parking available for registered delegates</p>
                  <p><strong>Public Transport:</strong> MyCiTi bus routes available, Uber/taxi services recommended</p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* FAQ Section */}
        <Card className="border-blue-200 mt-12">
          <CardHeader>
            <CardTitle className="text-2xl text-blue-900 text-center">Frequently Asked Questions</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="space-y-4">
                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">What is the conference language?</h3>
                  <p className="text-gray-700 text-sm">All sessions will be conducted in English. Simultaneous translation may be available for selected sessions.</p>
                </div>
                
                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">Is accommodation provided?</h3>
                  <p className="text-gray-700 text-sm">Accommodation is not included in registration fees, but we can provide recommendations for nearby hotels and guest houses.</p>
                </div>

                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">What should I bring?</h3>
                  <p className="text-gray-700 text-sm">Bring your laptop for workshops, business cards for networking, and comfortable clothing. All materials will be provided.</p>
                </div>
              </div>

              <div className="space-y-4">
                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">Are meals included?</h3>
                  <p className="text-gray-700 text-sm">Yes, all meals, coffee breaks, and the conference dinner are included in your registration fee.</p>
                </div>

                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">Can I get a certificate?</h3>
                  <p className="text-gray-700 text-sm">Yes, all registered participants will receive a certificate of attendance upon completion of the conference.</p>
                </div>

                <div>
                  <h3 className="font-semibold text-blue-800 mb-2">Is there a dress code?</h3>
                  <p className="text-gray-700 text-sm">Business casual attire is recommended. The conference dinner requires smart casual dress.</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}