import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Badge } from '@/components/ui/badge';
import { Users, CreditCard, CheckCircle, Calendar } from 'lucide-react';
import { toast } from 'sonner';

export default function Registration() {
  const [formData, setFormData] = useState({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    organization: '',
    position: '',
    country: '',
    registrationType: '',
    dietaryRequirements: '',
    accommodationNeeded: false,
    networkingEvents: true,
    agreeToTerms: false
  });

  const [isSubmitted, setIsSubmitted] = useState(false);

  const handleInputChange = (field: string, value: string | boolean) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.firstName || !formData.lastName || !formData.email || !formData.registrationType) {
      toast.error('Please fill in all required fields');
      return;
    }

    if (!formData.agreeToTerms) {
      toast.error('Please agree to the terms and conditions');
      return;
    }

    toast.success('Registration submitted successfully!');
    setIsSubmitted(true);
  };

  const registrationTypes = [
    {
      type: 'academic',
      title: 'Academic/Student',
      price: '$299',
      description: 'For university faculty, researchers, and students',
      features: ['All conference sessions', 'Conference materials', 'Meals and refreshments', 'Certificate of attendance']
    },
    {
      type: 'government',
      title: 'Government Official',
      price: '$399',
      description: 'For government employees and public sector workers',
      features: ['All conference sessions', 'Conference materials', 'Meals and refreshments', 'Certificate of attendance', 'Government networking session']
    },
    {
      type: 'industry',
      title: 'Industry Professional',
      price: '$499',
      description: 'For private sector and consulting professionals',
      features: ['All conference sessions', 'Conference materials', 'Meals and refreshments', 'Certificate of attendance', 'Industry networking session']
    },
    {
      type: 'international',
      title: 'International Delegate',
      price: '$349',
      description: 'Special rate for international participants',
      features: ['All conference sessions', 'Conference materials', 'Meals and refreshments', 'Certificate of attendance', 'Cultural program']
    }
  ];

  if (isSubmitted) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4 flex items-center justify-center">
        <Card className="max-w-2xl w-full border-green-200">
          <CardContent className="p-12 text-center">
            <CheckCircle className="w-16 h-16 text-green-600 mx-auto mb-6" />
            <h2 className="text-3xl font-bold text-green-800 mb-4">Registration Successful!</h2>
            <p className="text-lg text-gray-700 mb-6">
              Thank you for registering for IHEDG 2025. You will receive a confirmation email with payment instructions shortly.
            </p>
            <div className="bg-green-50 p-6 rounded-lg mb-6">
              <h3 className="font-semibold text-green-800 mb-2">Next Steps:</h3>
              <ul className="text-left text-green-700 space-y-2">
                <li>• Check your email for payment instructions</li>
                <li>• Complete payment within 7 days to secure your spot</li>
                <li>• Receive conference materials 2 weeks before the event</li>
                <li>• Join our WhatsApp group for updates</li>
              </ul>
            </div>
            <Button onClick={() => setIsSubmitted(false)} className="bg-blue-900 hover:bg-blue-800">
              Register Another Participant
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4">
      <div className="max-w-6xl mx-auto">
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-blue-900 mb-6">Conference Registration</h1>
          <p className="text-xl text-gray-700 max-w-3xl mx-auto">
            Join us for three days of insightful discussions on AI, digital governance, and higher education. 
            Register now to secure your place at this premier conference.
          </p>
          
          <div className="flex flex-wrap justify-center gap-4 mt-8">
            <div className="flex items-center space-x-2 text-blue-800">
              <Calendar className="w-5 h-5" />
              <span className="font-medium">April 8-10, 2025</span>
            </div>
            <div className="flex items-center space-x-2 text-blue-800">
              <Users className="w-5 h-5" />
              <span className="font-medium">Limited to 200 participants</span>
            </div>
          </div>
        </div>

        {/* Registration Types */}
        <div className="mb-12">
          <h2 className="text-2xl font-bold text-center text-blue-900 mb-8">Choose Your Registration Type</h2>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {registrationTypes.map((reg) => (
              <Card key={reg.type} className="border-blue-200 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
                <CardHeader className="text-center">
                  <CardTitle className="text-lg text-blue-900">{reg.title}</CardTitle>
                  <div className="text-3xl font-bold text-green-600">{reg.price}</div>
                  <p className="text-sm text-gray-600">{reg.description}</p>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-2">
                    {reg.features.map((feature, index) => (
                      <li key={index} className="text-sm text-gray-700 flex items-center">
                        <CheckCircle className="w-4 h-4 text-green-500 mr-2 flex-shrink-0" />
                        {feature}
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Registration Form */}
        <Card className="border-blue-200">
          <CardHeader>
            <CardTitle className="text-2xl text-blue-900 flex items-center gap-2">
              <CreditCard className="w-6 h-6" />
              Registration Form
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Personal Information */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Personal Information</h3>
                
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="firstName">First Name *</Label>
                    <Input
                      id="firstName"
                      value={formData.firstName}
                      onChange={(e) => handleInputChange('firstName', e.target.value)}
                      placeholder="Enter your first name"
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="lastName">Last Name *</Label>
                    <Input
                      id="lastName"
                      value={formData.lastName}
                      onChange={(e) => handleInputChange('lastName', e.target.value)}
                      placeholder="Enter your last name"
                      className="mt-1"
                      required
                    />
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="email">Email Address *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      placeholder="your.email@example.com"
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="phone">Phone Number</Label>
                    <Input
                      id="phone"
                      value={formData.phone}
                      onChange={(e) => handleInputChange('phone', e.target.value)}
                      placeholder="+1234567890"
                      className="mt-1"
                    />
                  </div>
                </div>
              </div>

              {/* Professional Information */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Professional Information</h3>
                
                <div>
                  <Label htmlFor="organization">Organization/Institution *</Label>
                  <Input
                    id="organization"
                    value={formData.organization}
                    onChange={(e) => handleInputChange('organization', e.target.value)}
                    placeholder="Your organization name"
                    className="mt-1"
                    required
                  />
                </div>

                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="position">Position/Title</Label>
                    <Input
                      id="position"
                      value={formData.position}
                      onChange={(e) => handleInputChange('position', e.target.value)}
                      placeholder="Your job title"
                      className="mt-1"
                    />
                  </div>
                  <div>
                    <Label htmlFor="country">Country *</Label>
                    <Input
                      id="country"
                      value={formData.country}
                      onChange={(e) => handleInputChange('country', e.target.value)}
                      placeholder="Your country"
                      className="mt-1"
                      required
                    />
                  </div>
                </div>
              </div>

              {/* Registration Type */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Registration Type *</h3>
                <RadioGroup 
                  value={formData.registrationType} 
                  onValueChange={(value) => handleInputChange('registrationType', value)}
                  className="grid md:grid-cols-2 gap-4"
                >
                  {registrationTypes.map((reg) => (
                    <div key={reg.type} className="flex items-center space-x-2 p-4 border rounded-lg hover:bg-blue-50">
                      <RadioGroupItem value={reg.type} id={reg.type} />
                      <div className="flex-1">
                        <Label htmlFor={reg.type} className="font-medium cursor-pointer">
                          {reg.title} - {reg.price}
                        </Label>
                        <p className="text-sm text-gray-600">{reg.description}</p>
                      </div>
                    </div>
                  ))}
                </RadioGroup>
              </div>

              {/* Additional Information */}
              <div className="space-y-4">
                <h3 className="text-lg font-semibold text-blue-800">Additional Information</h3>
                
                <div>
                  <Label htmlFor="dietary">Dietary Requirements</Label>
                  <Textarea
                    id="dietary"
                    value={formData.dietaryRequirements}
                    onChange={(e) => handleInputChange('dietaryRequirements', e.target.value)}
                    placeholder="Please specify any dietary restrictions or allergies"
                    className="mt-1"
                    rows={3}
                  />
                </div>

                <div className="space-y-3">
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="accommodation"
                      checked={formData.accommodationNeeded}
                      onCheckedChange={(checked) => handleInputChange('accommodationNeeded', checked as boolean)}
                    />
                    <Label htmlFor="accommodation" className="text-sm">
                      I need assistance with accommodation recommendations
                    </Label>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="networking"
                      checked={formData.networkingEvents}
                      onCheckedChange={(checked) => handleInputChange('networkingEvents', checked as boolean)}
                    />
                    <Label htmlFor="networking" className="text-sm">
                      I would like to participate in networking events
                    </Label>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="terms"
                      checked={formData.agreeToTerms}
                      onCheckedChange={(checked) => handleInputChange('agreeToTerms', checked as boolean)}
                    />
                    <Label htmlFor="terms" className="text-sm">
                      I agree to the conference terms and conditions *
                    </Label>
                  </div>
                </div>
              </div>

              {/* Submit Button */}
              <div className="flex justify-center pt-6">
                <Button type="submit" size="lg" className="bg-blue-900 hover:bg-blue-800 px-8">
                  Register Now
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>

        {/* Important Information */}
        <Card className="border-blue-200 mt-8">
          <CardContent className="p-6">
            <h3 className="text-lg font-semibold text-blue-800 mb-4">Important Information</h3>
            <div className="grid md:grid-cols-2 gap-6 text-sm text-gray-700">
              <div>
                <h4 className="font-medium text-blue-700 mb-2">Payment & Cancellation</h4>
                <ul className="space-y-1">
                  <li>• Payment due within 7 days of registration</li>
                  <li>• Full refund available until March 1, 2025</li>
                  <li>• 50% refund available until March 15, 2025</li>
                  <li>• No refund after March 15, 2025</li>
                </ul>
              </div>
              <div>
                <h4 className="font-medium text-blue-700 mb-2">What's Included</h4>
                <ul className="space-y-1">
                  <li>• Access to all conference sessions</li>
                  <li>• Conference materials and proceedings</li>
                  <li>• All meals and coffee breaks</li>
                  <li>• Certificate of attendance</li>
                  <li>• Networking events</li>
                </ul>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}